%% Fitting a transition model to the data: synthetic data
% by Martin H. Trauth, 13 Jan 2021 http://mres.uni-potsdam.de
clear, close all, clc

%% Example 1 Stepwise transition with noise
rng(0)
data(1:100,1) = 1 : 1 : 100;
data(1:45,2) = 0;
data(46:55,2) = 1/10 * data(46:55,1) - 4.5;
data(56:100,2) = 1;
data(:,2) = data(:,2) + 0.1*randn(size(data(:,2)));

%%
fo = fitoptions('Method','NonlinearLeastSquares',...
    'StartPoint',[0 1 50 1]);
ft = fittype(@(a,b,c,d,x) a+b*(1./(1+exp(-d*(x-c)))),...
    'Coeff',{'a','b','c','d'},...
    'Options',fo);
[f,fgof,fout] = fit(data(:,1),data(:,2),ft)

%%
clear rft
rft(:,1) = 1 : 1 : 100;
rft(:,2) = rampmht(rft(:,1),45,56);

%%
figure('Position',[10 1100 800 1200])
axes('XLim',[0 100],...
    'YLim',[-.2 1.2],...
    'Position',[0.1 0.8 0.8 0.15]), hold on
p = plot(f,data(:,1),data(:,2));
p(1).Color = [0.1 0.3 0.6];
p(1).Marker = 'Square';
p(1).MarkerSize = 6;
p(1).MarkerFaceColor = [0.1 0.3 0.6];
p(1).MarkerEdgeColor = [0.1 0.3 0.6];
p(2).Color = [0.7 0.3 0.1];
p(2).LineWidth = 1.5;
p(2).LineStyle = '-';

line(rft(:,1),rft(:,2),...                % Ramp Fit
    'Color',[0.7 0.3 0.1],...
    'LineWidth',1.5,...
    'LineStyle','-.')

lgd = legend('Data Points','Sigmoid Fit','Ramp Fit');
lgd.Location = 'southeast';
lgd.Box = 'Off';
lgd.FontSize = 12;
lgd.AutoUpdate = 'Off';
xlabel('t',...
    'FontSize',12)
ylabel('x',...
    'FontSize',12)

line([50.1 50.1],[0.7 0.9],...              % Inflection Sigmoid Fit
    'LineWidth',1.5,...
    'Color',[0.7 0.3 0.1])
text(50.1,1,'50.1',...
    'FontSize',18,...
    'Color',[0.7 0.3 0.1],...
    'HorizontalAlignment','Right')

line([50.5 50.5],[0.1 0.3],...              % Inflection Ramp Fit
    'LineWidth',1.5,...
    'LineStyle','-.',...
    'Color',[0.7 0.3 0.1])
text(50.5,0,'50.5',...
    'FontSize',18,...
    'Color',[0.7 0.3 0.1],...
    'HorizontalAlignment','Left')

%% Example 2 Rampwise transition with noise
rng(0)
data(1:100,1) = 1 : 1 : 100;
data(1:25,2) = 0;
data(26:75,2) = 1/50 * data(26:75,1) - 0.5;
data(76:100,2) = 1;
data(:,2) = data(:,2) + 0.1*randn(size(data(:,2)));

%%
fo = fitoptions('Method','NonlinearLeastSquares',...
    'StartPoint',[0 1 50 1]);
ft = fittype(@(a,b,c,d,x) a+b*(1./(1+exp(-d*(x-c)))),...
    'Coeff',{'a','b','c','d'},...
    'Options',fo);
[f,fgof,fout] = fit(data(:,1),data(:,2),ft)

%%
clear rft
rft(:,1) = 1 : 1 : 100;
rft(:,2) = rampmht(rft(:,1),30,73);

%%
axes('XLim',[0 100],...
    'YLim',[-.2 1.2],...
    'Position',[0.1 0.6 0.8 0.15]), hold on
p = plot(f,data(:,1),data(:,2));
p(1).Color = [0.1 0.3 0.6];
p(1).Marker = 'Square';
p(1).MarkerSize = 6;
p(1).MarkerFaceColor = [0.1 0.3 0.6];
p(1).MarkerEdgeColor = [0.1 0.3 0.6];
p(2).Color = [0.7 0.3 0.1];
p(2).LineWidth = 1.5;
p(2).LineStyle = '-';

line(rft(:,1),rft(:,2),...                % Ramp Fit
    'Color',[0.7 0.3 0.1],...
    'LineWidth',1.5,...
    'LineStyle','-.')

lgd = legend('Data Points','Sigmoid Fit','Ramp Fit');
lgd.Location = 'southeast';
lgd.Box = 'Off';
lgd.FontSize = 12;
lgd.AutoUpdate = 'Off';
xlabel('t',...
    'FontSize',12)
ylabel('x',...
    'FontSize',12)

line([51.4 51.4],[0.7 0.9],...              % Inflection Sigmoid Fit
    'LineWidth',1.5,...
    'Color',[0.7 0.3 0.1])
text(50.1,1,'51.4',...
    'FontSize',18,...
    'Color',[0.7 0.3 0.1],...
    'HorizontalAlignment','Right')

line([51.5 51.5],[0.1 0.3],...              % Inflection Ramp Fit
    'LineWidth',1.5,...
    'LineStyle','-.',...
    'Color',[0.7 0.3 0.1])
text(51.5,0,'51.5',...
    'FontSize',18,...
    'Color',[0.7 0.3 0.1],...
    'HorizontalAlignment','Left')

%% Example 3 Gradual transition with noise
rng(0)
data(1:100,1) = 1 : 1 : 100;
data(:,2) = 1/100*data(:,1);
data(:,2) = data(:,2) + 0.1*randn(size(data(:,2)));

%%
fo = fitoptions('Method','NonlinearLeastSquares',...
    'StartPoint',[0 1 50 1]);
ft = fittype(@(a,b,c,d,x) a+b*(1./(1+exp(-d*(x-c)))),...
    'Coeff',{'a','b','c','d'},...
    'Options',fo);
[f,fgof,fout] = fit(data(:,1),data(:,2),ft)

%%
clear rft
rft(:,1) = 1 : 1 : 100;
rft(:,2) = rampmht(rft(:,1),3,94);

%%
axes('XLim',[0 100],...
    'YLim',[-.2 1.2],...
    'Position',[0.1 0.4 0.8 0.15]), hold on
p = plot(f,data(:,1),data(:,2));
p(1).Color = [0.1 0.3 0.6];
p(1).Marker = 'Square';
p(1).MarkerSize = 6;
p(1).MarkerFaceColor = [0.1 0.3 0.6];
p(1).MarkerEdgeColor = [0.1 0.3 0.6];
p(2).Color = [0.7 0.3 0.1];
p(2).LineWidth = 1.5;
p(2).LineStyle = '-';

line(rft(:,1),rft(:,2),...                % Ramp Fit
    'Color',[0.7 0.3 0.1],...
    'LineWidth',1.5,...
    'LineStyle','-.')

lgd = legend('Data Points','Sigmoid Fit','Ramp Fit');
lgd.Location = 'southeast';
lgd.Box = 'Off';
lgd.FontSize = 12;
lgd.AutoUpdate = 'Off';
xlabel('t',...
    'FontSize',12)
ylabel('x',...
    'FontSize',12)

line([57.1 57.1],[0.7 0.9],...              % Inflection Sigmoid Fit
    'LineWidth',1.5,...
    'Color',[0.7 0.3 0.1])
text(57.1,1,'57.1',...
    'FontSize',18,...
    'Color',[0.7 0.3 0.1],...
    'HorizontalAlignment','Right')

line([48.5 48.5],[0.1 0.3],...              % Inflection Ramp Fit
    'LineWidth',1.5,...
    'LineStyle','-.',...
    'Color',[0.7 0.3 0.1])
text(48.5,0,'48.5',...
    'FontSize',18,...
    'Color',[0.7 0.3 0.1],...
    'HorizontalAlignment','Left')

%% Example 4 Gradual change with noise but with a large gap
rng(0)
data(1:100,1) = 1 : 1 : 100;
data(:,2) = 1/100*data(:,1);
data(:,2) = data(:,2) + 0.1*randn(size(data(:,2)));
data(46:65,:) = [];

%%
fo = fitoptions('Method','NonlinearLeastSquares',...
    'StartPoint',[0 1 50 1]);
ft = fittype(@(a,b,c,d,x) a+b*(1./(1+exp(-d*(x-c)))),...
    'Coeff',{'a','b','c','d'},...
    'Options',fo);
[f,fgof,fout] = fit(data(:,1),data(:,2),ft)

%%
clear rft
rft(:,1) = 1 : 1 : 100;
rft(:,2) = rampmht(rft(:,1),3,94);

%%
axes('XLim',[0 100],...
    'YLim',[-.2 1.2],...
    'Position',[0.1 0.2 0.8 0.15]), hold on
p = plot(f,data(:,1),data(:,2));
p(1).Color = [0.1 0.3 0.6];
p(1).Marker = 'Square';
p(1).MarkerSize = 6;
p(1).MarkerFaceColor = [0.1 0.3 0.6];
p(1).MarkerEdgeColor = [0.1 0.3 0.6];
p(2).Color = [0.7 0.3 0.1];
p(2).LineWidth = 1.5;
p(2).LineStyle = '-';

line(rft(:,1),rft(:,2),...                % Ramp Fit
    'Color',[0.7 0.3 0.1],...
    'LineWidth',1.5,...
    'LineStyle','-.')

lgd = legend('Data Points','Sigmoid Fit','Ramp Fit');
lgd.Location = 'southeast';
lgd.Box = 'Off';
lgd.FontSize = 12;
lgd.AutoUpdate = 'Off';
xlabel('t',...
    'FontSize',12)
ylabel('x',...
    'FontSize',12)

line([62.9 62.9],[0.7 0.9],...              % Inflection Sigmoid Fit
    'LineWidth',1.5,...
    'Color',[0.7 0.3 0.1])
text(62.9,1,'62.9',...
    'FontSize',18,...
    'Color',[0.7 0.3 0.1],...
    'HorizontalAlignment','Right')

line([48.5 48.5],[0.1 0.3],...              % Inflection Ramp Fit
    'LineWidth',1.5,...
    'LineStyle','-.',...
    'Color',[0.7 0.3 0.1])
text(48.5,0,'48.5',...
    'FontSize',18,...
    'Color',[0.7 0.3 0.1],...
    'HorizontalAlignment','Left')

%%
print -dpng -r300 figure_synthetic_single_vs3_sigmoid.png




